#pragma once
#include "graph.hpp"
#include "tarjan.hpp"
#include "dinic.hpp"
#include "atoms.hpp"
#include "cycles.hpp"
#include <vector>
#include <algorithm>

const int INF = 1e9;

inline int cyclic_edge_connectivity(const Graph& g,
    std::vector<std::vector<int>>& min_cuts)
{
    min_cuts.clear();

    // Step 1: check for bridges
    BridgeFinder bf(g);
    bf.run();
    if (!bf.bridges.empty()) {
        // Check if both sides of bridge have cycles
        for (int e : bf.bridges) {
            std::vector<bool> removed(g.edges.size(), false);
            removed[e] = true;

            // BFS/DFS to find components
            std::vector<bool> vis(g.n, false);
            std::vector<std::vector<int>> components;
            for (int i = 0; i < g.n; ++i) if (!vis[i]) {
                std::vector<int> comp;
                std::vector<int> stack = { i };
                vis[i] = true;
                while (!stack.empty()) {
                    int v = stack.back(); stack.pop_back();
                    comp.push_back(v);
                    for (auto edge : g.adj[v])
                        if (!removed[edge.id] && !vis[edge.to]) {
                            vis[edge.to] = true;
                            stack.push_back(edge.to);
                        }
                }
                components.push_back(comp);
            }

            int cyclic_components = 0;
            for (auto& comp : components)
                if (atoms::component_has_cycle(g, comp))
                    cyclic_components++;

            if (cyclic_components >= 2)
                min_cuts.push_back({ e });
        }

        if (!min_cuts.empty())
            return (int)min_cuts[0].size();
    }

    // Step 2: enumerate all cycles
    auto cycles = enumerate_cycles(g);
    if (cycles.size() < 2) {
        // Not enough cycles to have a cyclic edge cut
        return 0;
    }

    int best = INF;

    // Step 3: try all pairs of cycles
    for (size_t i = 0; i < cycles.size(); ++i)
        for (size_t j = i + 1; j < cycles.size(); ++j) {
            auto& C1 = cycles[i];
            auto& C2 = cycles[j];

            // Flow network:
            Dinic D(g.n + 2);
            int S = g.n, T = g.n + 1;
            for (auto& e : g.edges) {
                int u = e.first;
                int v = e.second;
                D.add_edge(u, v, 1);
                D.add_edge(v, u, 1);
            }


            for (int v : C1) D.add_edge(S, v, INF);
            for (int v : C2) D.add_edge(v, T, INF);

            int f = D.max_flow(S, T);
            if (f > 0) {
                if (f < best) {
                    best = f;
                    min_cuts.clear();
                }
                if (f == best)
                    min_cuts.push_back({});
            }
        }

    if (best == INF) return 0; // (no cyclic cut found)
    return best;
}
